/***************************************************************************

Mappy memory map (preliminary)

CPU #1:
0000-07ff video RAM
0800-0fff color RAM
1000-177f RAM
1780-17ff sprite data 1 (sprite number & color)
1800-1f7f RAM
1f80-1fff sprite data 2 (x, y position)
2000-277f RAM
2780-27ff sprite data 3 (high bit of y, flip flags, double-size flags)
3800-3fff scroll register map
4040-43ff RAM shared with CPU #2
4800-480f custom I/O chip #1
4810-481f custom I/O chip #2
5002-5003 IRQ enable
500a-500b CPU #2 enable
8000      watchdog timer
a000-ffff ROM

CPU #2:
0000-0040 sound registers
0040-03ff RAM shared with CPU #1
2000-2001 IRQ enable
2006-2007 sound enable
e000-ffff ROM

Interrupts:
CPU #1 IRQ generated by VBLANK
CPU #2 IRQ generated by VBLANK

***************************************************************************/

#include "driver.h"
#include "vidhrdw/generic.h"

/* machine driver data & functions */
extern unsigned char *mappy_sharedram;
extern unsigned char *mappy_customio_1,*mappy_customio_2;
int mappy_init_machine(const char *gamename);
int mappy_sharedram_r(int offset);
int mappy_sharedram_r2(int offset);
int digdug2_sharedram_r2(int offset);
int mappy_cpu1ram_r(int offset);
int digdug2_cpu1ram_r(int offset);
void mappy_sharedram_w(int offset,int data);
void mappy_customio_w_1(int offset,int data);
void mappy_customio_w_2(int offset,int data);
int mappy_customio_r_1(int offset);
int mappy_customio_r_2(int offset);
int mappy_interrupt_1(void);
int mappy_interrupt_2(void);
void mappy_interrupt_enable_1_w(int offset,int data);
void mappy_interrupt_enable_2_w(int offset,int data);
void mappy_cpu_enable_w(int offset,int data);

/* video driver data & functions */
int mappy_vh_start(void);
void mappy_vh_stop(void);
void mappy_vh_screenrefresh(struct osd_bitmap *bitmap);
void mappy_videoram_w(int offset,int data);
void mappy_colorram_w(int offset,int data);
void mappy_scroll_w(int offset,int data);
void mappy_vh_convert_color_prom(unsigned char *palette, unsigned char *colortable,const unsigned char *color_prom);

/* sound driver data & functions */
extern unsigned char *mappy_soundregs;
void mappy_sh_update(void);
void mappy_sound_w(int offset,int data);
void mappy_sound_enable_w(int offset,int data);


/* CPU 1 read addresses */
static struct MemoryReadAddress mappy_readmem_cpu1[] =
{
	{ 0x8000, 0xffff, MRA_ROM },                                 /* ROM code */
	{ 0x4040, 0x43ff, MRA_RAM, &mappy_sharedram },               /* shared RAM with the sound CPU */
	{ 0x4800, 0x480f, mappy_customio_r_1, &mappy_customio_1 },   /* custom I/O chip #1 interface */
	{ 0x4810, 0x481f, mappy_customio_r_2, &mappy_customio_2 },   /* custom I/O chip #2 interface */
	{ 0x0000, 0x9fff, mappy_cpu1ram_r },                         /* RAM everywhere else */

	{ -1 }	/* end of table */
};


static struct MemoryReadAddress digdug2_readmem_cpu1[] =
{
	{ 0x8000, 0xffff, MRA_ROM },                                 /* ROM code */
	{ 0x4040, 0x43ff, MRA_RAM, &mappy_sharedram },               /* shared RAM with the sound CPU */
	{ 0x4800, 0x480f, mappy_customio_r_1, &mappy_customio_1 },   /* custom I/O chip #1 interface */
	{ 0x4810, 0x481f, mappy_customio_r_2, &mappy_customio_2 },   /* custom I/O chip #2 interface */
	{ 0x0000, 0x9fff, digdug2_cpu1ram_r },                       /* RAM everywhere else */

	{ -1 }	/* end of table */
};


/* CPU 1 write addresses */
static struct MemoryWriteAddress writemem_cpu1[] =
{
	{ 0x1000, 0x177f, MWA_RAM },                                 /* general RAM, area 1 */
	{ 0x1800, 0x1f7f, MWA_RAM },                                 /* general RAM, area 2 */
	{ 0x2000, 0x277f, MWA_RAM },                                 /* general RAM, area 3 */
	{ 0x4040, 0x43ff, MWA_RAM },                                 /* shared RAM with the sound CPU */
	{ 0x0000, 0x07ff, mappy_videoram_w, &videoram, &videoram_size },/* video RAM */
	{ 0x0800, 0x0fff, mappy_colorram_w, &colorram },             /* color RAM */
	{ 0x1780, 0x17ff, MWA_RAM, &spriteram, &spriteram_size },    /* sprite RAM, area 1 */
	{ 0x1f80, 0x1fff, MWA_RAM, &spriteram_2 },                   /* sprite RAM, area 2 */
	{ 0x2780, 0x27ff, MWA_RAM, &spriteram_3 },                   /* sprite RAM, area 3 */
	{ 0x3800, 0x3fff, mappy_scroll_w },                          /* scroll registers */
	{ 0x4800, 0x480f, mappy_customio_w_1 },                      /* custom I/O chip #1 interface */
	{ 0x4810, 0x481f, mappy_customio_w_2 },                      /* custom I/O chip #2 interface */
	{ 0x5002, 0x5003, mappy_interrupt_enable_1_w },              /* interrupt enable */
	{ 0x500a, 0x500b, mappy_cpu_enable_w },                      /* sound CPU enable */
	{ 0x8000, 0x8000, MWA_NOP },                                 /* watchdog timer */
	{ 0x8000, 0xffff, MWA_ROM },                                 /* ROM code */

	{ -1 }	/* end of table */
};


/* CPU 2 read addresses */
static struct MemoryReadAddress mappy_readmem_cpu2[] =
{
	{ 0xe000, 0xffff, MRA_ROM },                                 /* ROM code */
	{ 0x0040, 0x03ff, mappy_sharedram_r2 },                      /* shared RAM with the main CPU */

	{ -1 }	/* end of table */
};


static struct MemoryReadAddress digdug2_readmem_cpu2[] =
{
	{ 0xe000, 0xffff, MRA_ROM },                                 /* ROM code */
	{ 0x0040, 0x03ff, digdug2_sharedram_r2 },                    /* shared RAM with the main CPU */

	{ -1 }	/* end of table */
};


/* CPU 2 write addresses */
static struct MemoryWriteAddress writemem_cpu2[] =
{
	{ 0x0040, 0x03ff, mappy_sharedram_w },                       /* shared RAM with the main CPU */
	{ 0x0000, 0x003f, mappy_sound_w, &mappy_soundregs },         /* sound control registers */
	{ 0x2000, 0x2001, mappy_interrupt_enable_2_w },              /* interrupt enable */
	{ 0x2006, 0x2007, mappy_sound_enable_w },                    /* sound enable */
	{ 0xe000, 0xffff, MWA_ROM },                                 /* ROM code */

	{ -1 }	/* end of table */
};


/* input from the outside world */
static struct InputPort mappy_input_ports[] =
{
	{	/* DSW1 */
		0x00,
		{ 0, 0, 0, 0, 0, 0, OSD_KEY_F1, 0 },
		{ 0, 0, 0, 0, 0, 0, 0, 0 }
	},
	{	/* DSW2 */
		0x00,
		{ 0, 0, 0, 0, 0, 0, 0, 0 },
		{ 0, 0, 0, 0, 0, 0, 0, 0 }
	},
	{	/* IN0 */
		0x00,
		{ 0, OSD_KEY_RIGHT, 0, OSD_KEY_LEFT, 0, OSD_KEY_CONTROL, 0, 0 },
		{ 0, OSD_JOY_RIGHT, 0, OSD_JOY_LEFT, 0, OSD_JOY_FIRE, 0, 0 },
	},
	{	/* IN1 */
		0x00,
		{ OSD_KEY_3, 0, 0, 0, OSD_KEY_1, OSD_KEY_2, 0, 0 },
		{ 0, 0, 0, 0, 0, 0, 0, 0 },
	},
	{	/* DSW3 */
		0x00,
		{ 0, 0, 0, 0, 0, 0, 0, 0 },
		{ 0, 0, 0, 0, 0, 0, 0, 0 }
	},
	{ -1 }	/* end of table */
};


static struct InputPort digdug2_input_ports[] =
{
	{	/* DSW1 */
		0x00,
		{ 0, 0, 0, 0, 0, 0, OSD_KEY_F1, 0 },
		{ 0, 0, 0, 0, 0, 0, 0, 0 }
	},
	{	/* DSW2 */
		0x00,
		{ 0, 0, 0, 0, 0, 0, 0, 0 },
		{ 0, 0, 0, 0, 0, 0, 0, 0 }
	},
	{	/* IN0 */
		0x00,
		{ OSD_KEY_UP, OSD_KEY_RIGHT, OSD_KEY_DOWN, OSD_KEY_LEFT, 0, OSD_KEY_CONTROL, 0, 0 },
		{ OSD_JOY_UP, OSD_JOY_RIGHT, OSD_JOY_DOWN, OSD_JOY_LEFT, 0, OSD_JOY_FIRE, 0, 0 },
	},
	{	/* IN1 */
		0x00,
		{ OSD_KEY_3, 0, 0, 0, OSD_KEY_1, OSD_KEY_2, 0, 0 },
		{ 0, 0, 0, 0, 0, 0, 0, 0 },
	},
	{	/* IN2/DSW3 */
		0x00,
		{ 0, 0, 0, 0, 0, OSD_KEY_ALT, 0, 0 },
		{ 0, 0, 0, 0, 0, OSD_JOY_FIRE2, 0, 0 }
	},
	{ -1 }	/* end of table */
};


static struct TrakPort trak_ports[] =
{
        { -1 }
};


/* key descriptions and links */
static struct KEYSet mappy_keys[] =
{
	{ 2, 3, "MOVE LEFT"  },
	{ 2, 1, "MOVE RIGHT" },
	{ 2, 5, "OPEN DOOR"  },
	{ -1 }
};


static struct KEYSet digdug2_keys[] =
{
	{ 2, 3, "MOVE LEFT"  },
	{ 2, 1, "MOVE RIGHT" },
	{ 2, 0, "MOVE UP"  },
	{ 2, 2, "MOVE DOWN"  },
	{ 2, 5, "PUMP"  },
	{ 4, 5, "DRILL"  },
	{ -1 }
};


/* dipswitch menu data */
static struct DSW mappy_dsw[] =
{
	{ 1, 0xc0, "LIVES",      { "3", "5", "1", "2" }, 0, },
	{ 1, 0x38, "BONUS",      { "20K 70K", "20K 60K", "20K 80K", "30K 100K", "20K NONE", "20K 70K 70K", "20K 80K 80K", "NONE" }, 0 },
	{ 0, 0x03, "DIFFICULTY", { "EASY", "MEDIUM", "HARD", "HARDEST" }, 0 },
	{ 0, 0x20, "DEMO SOUND", { "ON", "OFF" }, 1 },
	{ -1 }
};


static struct DSW digdug2_dsw[] =
{
	{ 0, 0x20, "LIVES",        { "3", "5" }, 0, },
	{ 4, 0x04, "LEVEL SELECT", { "OFF", "ON" }, 0 },
	{ 4, 0x03, "EXTEND",       { "A", "C", "B", "D" }, 0 },
	{ -1 }
};


/* layout of the 8x8x2 character data */
static struct GfxLayout charlayout =
{
	8,8,	         /* 8*8 characters */
	256,	         /* 256 characters */
	2,             /* 2 bits per pixel */
	{ 0, 4 },      /* the two bitplanes for 4 pixels are packed into one byte */
	{ 7*8, 6*8, 5*8, 4*8, 3*8, 2*8, 1*8, 0*8 },   /* characters are rotated 90 degrees */
	{ 8*8+0, 8*8+1, 8*8+2, 8*8+3, 0, 1, 2, 3 },   /* bits are packed in groups of four */
	16*8	       /* every char takes 16 bytes */
};


/* layout of the 16x16x4 sprite data */
static struct GfxLayout mappy_spritelayout =
{
	16,16,	     /* 16*16 sprites */
	128,	        /* 128 sprites */
	4,	           /* 4 bits per pixel */
	{ 0, 4, 8192*8, 8192*8+4 },	/* the two bitplanes for 4 pixels are packed into one byte */
	{ 39 * 8, 38 * 8, 37 * 8, 36 * 8, 35 * 8, 34 * 8, 33 * 8, 32 * 8,
			7 * 8, 6 * 8, 5 * 8, 4 * 8, 3 * 8, 2 * 8, 1 * 8, 0 * 8 },
	{ 0, 1, 2, 3, 8*8, 8*8+1, 8*8+2, 8*8+3, 16*8+0, 16*8+1, 16*8+2, 16*8+3,
			24*8+0, 24*8+1, 24*8+2, 24*8+3 },
	64*8	/* every sprite takes 64 bytes */
};


static struct GfxLayout digdug2_spritelayout =
{
	16,16,	     /* 16*16 sprites */
	256,	        /* 256 sprites */
	4,	           /* 4 bits per pixel */
	{ 0, 4, 16384*8, 16384*8+4 },	/* the two bitplanes for 4 pixels are packed into one byte */
	{ 39 * 8, 38 * 8, 37 * 8, 36 * 8, 35 * 8, 34 * 8, 33 * 8, 32 * 8,
			7 * 8, 6 * 8, 5 * 8, 4 * 8, 3 * 8, 2 * 8, 1 * 8, 0 * 8 },
	{ 0, 1, 2, 3, 8*8, 8*8+1, 8*8+2, 8*8+3, 16*8+0, 16*8+1, 16*8+2, 16*8+3,
			24*8+0, 24*8+1, 24*8+2, 24*8+3 },
	64*8	/* every sprite takes 64 bytes */
};


/* pointers to the appropriate memory locations and their associated decode structs */
static struct GfxDecodeInfo mappy_gfxdecodeinfo[] =
{
	{ 1, 0x0000, &charlayout,      0, 64 },
	{ 1, 0x1000, &mappy_spritelayout, 64*4, 16 },
	{ -1 } /* end of array */
};


static struct GfxDecodeInfo digdug2_gfxdecodeinfo[] =
{
	{ 1, 0x0000, &charlayout,      0, 64 },
	{ 1, 0x1000, &digdug2_spritelayout, 64*4, 16 },
	{ -1 } /* end of array */
};


static unsigned char mappy_color_prom[] =
{
	/* palette */
	0x00,0xd9,0xa4,0x5d,0x36,0x80,0xa7,0x26,0x68,0x06,0x1d,0x66,0xa8,0xde,0xf6,0x00,
	0x00,0x38,0x36,0x06,0x66,0x00,0x5d,0xf0,0xb7,0x26,0x00,0x00,0xe7,0xd9,0xf6,0x00,
   /* characters */
	0x0f,0x01,0x04,0x08,0x0f,0x0d,0x04,0x08,0x0f,0x02,0x04,0x08,0x0f,0x03,0x04,0x08,
	0x0f,0x0d,0x04,0x08,0x0f,0x03,0x04,0x08,0x0f,0x01,0x04,0x08,0x0f,0x0e,0x00,0x00,
	0x00,0x00,0x00,0x00,0x0f,0x08,0x00,0x00,0x0f,0x06,0x04,0x08,0x0f,0x09,0x00,0x00,
	0x0f,0x0c,0x00,0x00,0x0f,0x07,0x00,0x00,0x0f,0x09,0x07,0x00,0x0f,0x02,0x0e,0x0d,
	0x0f,0x03,0x01,0x01,0x0f,0x03,0x0d,0x01,0x0f,0x09,0x02,0x01,0x0f,0x0c,0x03,0x01,
	0x0f,0x0c,0x0d,0x01,0x0f,0x09,0x03,0x01,0x0f,0x07,0x01,0x01,0x0f,0x07,0x0e,0x01,
	0x0f,0x03,0x0e,0x01,0x0f,0x0d,0x0e,0x02,0x0f,0x0d,0x04,0x08,0x0f,0x00,0x09,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	/* sprites */
	0x00,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,
	0x00,0x08,0x02,0x03,0x07,0x05,0x06,0x07,0x09,0x01,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,
	0x00,0x09,0x02,0x03,0x08,0x05,0x06,0x07,0x01,0x08,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,
	0x00,0x01,0x02,0x03,0x04,0x05,0x0f,0x07,0x08,0x0f,0x0a,0x0b,0x0c,0x0d,0x0e,0x0f,
	0x00,0x05,0x02,0x09,0x09,0x05,0x00,0x00,0x02,0x09,0x00,0x00,0x00,0x05,0x00,0x00,
	0x00,0x02,0x00,0x02,0x05,0x00,0x02,0x00,0x02,0x00,0x05,0x00,0x00,0x05,0x00,0x00,
	0x00,0x05,0x02,0x07,0x07,0x05,0x00,0x07,0x02,0x09,0x00,0x02,0x09,0x05,0x00,0x00,
	0x00,0x02,0x00,0x02,0x05,0x00,0x02,0x00,0x02,0x00,0x05,0x05,0x02,0x05,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
	0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f,0x0f
};


static unsigned char digdug2_color_prom[] =
{
	/* palette */
	0x00,0x07,0x58,0x3f,0xf6,0xaf,0xe7,0xeb,0xd1,0x6e,0x1c,0x22,0x28,0x8c,0x2f,0x00,
	0x00,0xf6,0xeb,0x06,0x3f,0x39,0xa6,0xe7,0x43,0xaf,0x6e,0x1c,0x22,0x28,0xc0,0x00,
   /* characters */
	0x00,0x01,0x00,0x00,0x0e,0x0e,0x0b,0x0e,0x0e,0x0e,0x0a,0x0b,0x0d,0x0c,0x0e,0x0b,
	0x0e,0x0b,0x0e,0x0b,0x0d,0x0c,0x0d,0x0a,0x0d,0x0c,0x0e,0x0b,0x0d,0x0c,0x0b,0x0e,
	0x00,0x09,0x01,0x02,0x0d,0x0c,0x0e,0x0b,0x00,0x0d,0x0a,0x0b,0x00,0x04,0x00,0x00,
	0x0d,0x0c,0x0e,0x0b,0x00,0x03,0x01,0x02,0x00,0x0e,0x0a,0x0b,0x0e,0x01,0x02,0x00,
	0x0d,0x0c,0x00,0x0b,0x0d,0x0c,0x0a,0x0b,0x0d,0x0c,0x00,0x0b,0x0d,0x0c,0x00,0x0b,
	0x0d,0x0c,0x00,0x0b,0x0e,0x0a,0x0e,0x0a,0x0d,0x0c,0x00,0x0b,0x0d,0x0c,0x0b,0x0b,
	0x0d,0x0c,0x00,0x0b,0x0d,0x0c,0x00,0x0b,0x0d,0x0c,0x0e,0x0b,0x0d,0x0c,0x0b,0x0b,
	0x0d,0x0c,0x00,0x0b,0x0b,0x0b,0x0a,0x0a,0x0a,0x0e,0x0e,0x0e,0x0d,0x0b,0x0a,0x0a,
	0x0d,0x0c,0x00,0x0e,0x0d,0x0c,0x00,0x0e,0x0d,0x0c,0x0b,0x0e,0x0d,0x0c,0x00,0x0e,
	0x0d,0x0c,0x00,0x0b,0x0d,0x0c,0x0d,0x0b,0x0d,0x0c,0x0a,0x0e,0x0d,0x0c,0x0b,0x0a,
	0x0d,0x0c,0x0b,0x0e,0x0d,0x0c,0x0e,0x0e,0x0d,0x0b,0x0b,0x0a,0x0d,0x0b,0x0b,0x0a,
	0x0d,0x0c,0x0d,0x0e,0x0d,0x0c,0x0a,0x0d,0x0e,0x0e,0x0b,0x0b,0x0e,0x0e,0x0a,0x0a,
	0x0d,0x08,0x00,0x0a,0x0e,0x0b,0x0b,0x0a,0x0b,0x0b,0x0b,0x0a,0x0a,0x0a,0x0b,0x0b,
	0x0b,0x0b,0x0b,0x0a,0x0e,0x0b,0x0b,0x0a,0x0e,0x0b,0x0e,0x0a,0x0a,0x0b,0x0b,0x0a,
	0x0a,0x0a,0x0b,0x0a,0x0e,0x0b,0x0a,0x0a,0x0d,0x0c,0x0b,0x0a,0x0d,0x0c,0x0a,0x0a,
	0x0d,0x0c,0x0e,0x0a,0x0d,0x0b,0x0e,0x0a,0x0b,0x0a,0x0e,0x0e,0x0e,0x0e,0x0a,0x0e,
	/* sprites */
	0x0f,0x01,0x0f,0x01,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x00,
	0x0f,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x00,
	0x08,0x04,0x04,0x04,0x04,0x04,0x04,0x04,0x03,0x03,0x03,0x03,0x03,0x03,0x03,0x03,
	0x0f,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x00,
	0x0f,0x01,0x02,0x03,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x00,
	0x0f,0x04,0x07,0x04,0x07,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x07,0x0d,0x04,
	0x0f,0x07,0x04,0x07,0x04,0x0a,0x09,0x08,0x07,0x06,0x05,0x04,0x03,0x07,0x0d,0x04,
	0x0f,0x04,0x07,0x04,0x07,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x07,0x0d,0x04,
	0x0f,0x07,0x04,0x07,0x04,0x0a,0x09,0x08,0x07,0x06,0x05,0x04,0x03,0x07,0x0d,0x04,
	0x08,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0x08,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0x08,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0x0f,0x0f,0x08,0x08,0x04,0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x0c,0x0d,0x0e,0x00,
	0x0f,0x07,0x0f,0x07,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0x01,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,
	0x07,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff,0xff
};


/* waveforms for the audio hardware */
static unsigned char mappy_samples[8*32] =
{
	0x68,0x68,0x68,0x68,0x68,0x68,0x68,0x68,0x68,0x68,0x68,0x68,0x68,0x68,0x68,0x68,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,

	0x88,0x88,0x98,0x98,0xa8,0xa8,0xb8,0xb8,0xc8,0xc8,0xd8,0xd8,0xe8,0xe8,0xf8,0xf8,
	0x08,0x08,0x18,0x18,0x28,0x28,0x38,0x38,0x48,0x48,0x58,0x58,0x68,0x68,0x78,0x78,

	0x88,0xa8,0xb8,0xc8,0xe8,0xe8,0xc8,0xd8,0x18,0x28,0x28,0x08,0x28,0xf8,0x28,0x18,
	0xf8,0xf8,0x18,0xe8,0x08,0xe8,0xd8,0xe8,0x28,0x48,0x28,0x28,0x48,0x58,0x68,0x78,

	0x38,0x58,0x68,0x58,0x48,0x28,0x08,0x08,0x08,0x28,0x48,0x58,0x68,0x58,0x38,0x08,
	0xc8,0xa8,0x98,0xa8,0xb8,0xd8,0xf8,0xf8,0xf8,0xd8,0xb8,0xa8,0x98,0xa8,0xc8,0xf8,

	0xf8,0x28,0x48,0x58,0x68,0x58,0x48,0x28,0xf8,0xc8,0xa8,0x98,0x88,0x98,0xa8,0xc8,
	0xf8,0x38,0x58,0x68,0x58,0x38,0xf8,0xb8,0x98,0x88,0x98,0xb8,0xf8,0x68,0xf8,0x88,

	0x28,0x48,0x48,0x28,0xf8,0xf8,0x08,0x38,0x58,0x68,0x58,0x28,0xe8,0xd8,0xd8,0xf8,
	0x18,0x18,0x08,0xc8,0x98,0x88,0x98,0xb8,0xe8,0xf8,0xf8,0xc8,0xa8,0xa8,0xc8,0xf8,

	0xf8,0x68,0x48,0x18,0x48,0x68,0x28,0xf8,0x48,0x78,0x58,0x08,0x28,0x38,0xf8,0xa8,
	0x08,0x58,0x18,0xc8,0xd8,0xf8,0xa8,0x88,0xb8,0x08,0xd8,0x98,0xb8,0xe8,0xb8,0x98,

	0x68,0x68,0xf8,0xf8,0x68,0x68,0xf8,0xf8,0x68,0x68,0xf8,0xf8,0x68,0x68,0xf8,0xf8,
	0x88,0x88,0xf8,0xf8,0x88,0x88,0xf8,0xf8,0x88,0x88,0xf8,0xf8,0x88,0x88,0xf8,0xf8
};


static unsigned char digdug2_samples[8*32] =
{
	0xf8,0x18,0x28,0x38,0x48,0x58,0x58,0x68,0x68,0x68,0x58,0x58,0x48,0x38,0x28,0x18,
	0xf8,0xd8,0xc8,0xb8,0xa8,0x98,0x98,0x88,0x88,0x88,0x98,0x98,0xa8,0xb8,0xc8,0xd8,

	0xf8,0x18,0x28,0x38,0xf8,0x58,0x58,0xf8,0x68,0xf8,0x58,0x58,0xf8,0x38,0x28,0x18,
	0xf8,0xd8,0xf8,0xb8,0xf8,0x98,0xf8,0x88,0xf8,0x88,0xf8,0x98,0xf8,0xb8,0xf8,0xd8,

	0x68,0x68,0x68,0x68,0x68,0x68,0x68,0x68,0x68,0x68,0x68,0x68,0x68,0x68,0x68,0x68,
	0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,0x88,

	0x38,0x58,0x68,0x58,0x48,0x28,0x08,0x08,0x08,0x28,0x48,0x58,0x68,0x58,0x38,0x08,
	0xc8,0xa8,0x98,0xa8,0xb8,0xd8,0xf8,0xf8,0xf8,0xd8,0xb8,0xa8,0x98,0xa8,0xc8,0xf8,

	0xf8,0x28,0x48,0x58,0x68,0x58,0x48,0x28,0xf8,0xc8,0xa8,0x98,0x88,0x98,0xa8,0xc8,
	0xf8,0x38,0x58,0x68,0x58,0x38,0xf8,0xb8,0x98,0x88,0x98,0xb8,0xf8,0x68,0xf8,0x88,

	0xf8,0x68,0x48,0x18,0x48,0x68,0x28,0xf8,0x48,0x78,0x58,0x08,0x28,0x38,0xf8,0xa8,
	0x08,0x58,0x18,0xc8,0xd8,0xf8,0xa8,0x88,0xb8,0x08,0xd8,0x98,0xb8,0xe8,0xb8,0x98,

	0xf8,0x28,0x48,0x48,0x68,0x68,0x48,0x48,0x68,0x68,0x48,0x18,0xf8,0xf8,0xd8,0xd8,
	0xf8,0x18,0x18,0xf8,0xf8,0xd8,0xa8,0x88,0x88,0xa8,0xa8,0x88,0x88,0xa8,0xa8,0xb8,

	0x38,0x48,0x58,0x68,0x28,0x38,0x38,0x48,0x28,0x38,0xf8,0x18,0x28,0x38,0x48,0x48,
	0x98,0xa8,0xa8,0xb8,0xc8,0xd8,0xf8,0xb8,0x98,0xa8,0xb8,0xb8,0xc8,0x88,0x98,0xa8
};


/* the machine driver: 2 6809s running at 1MHz */
static struct MachineDriver mappy_machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_M6809,
			1100000,			/* 1.1 Mhz */
			0,
			mappy_readmem_cpu1,writemem_cpu1,0,0,
			mappy_interrupt_1,1
		},
		{
			CPU_M6809,
			1100000,			/* 1.1 Mhz */
			2,	/* memory region #2 */
			mappy_readmem_cpu2,writemem_cpu2,0,0,
			mappy_interrupt_2,1
		}
	},
	60,
	mappy_init_machine,

	/* video hardware */
	28*8, 36*8, { 0*8, 28*8-1, 0*8, 36*8-1 },
	mappy_gfxdecodeinfo,
	32,64*4+16*16,
	mappy_vh_convert_color_prom,

	0,
	mappy_vh_start,
	mappy_vh_stop,
	mappy_vh_screenrefresh,

	/* sound hardware */
	mappy_samples,
	0,
	0,
	0,
	mappy_sh_update
};



/* the machine driver: 2 6809s running at 1MHz */
static struct MachineDriver digdug2_machine_driver =
{
	/* basic machine hardware */
	{
		{
			CPU_M6809,
			1600000,			/* 1.6 Mhz */
			0,
			digdug2_readmem_cpu1,writemem_cpu1,0,0,
			mappy_interrupt_1,1
		},
		{
			CPU_M6809,
			1600000,			/* 1.6 Mhz */
			2,	/* memory region #2 */
			digdug2_readmem_cpu2,writemem_cpu2,0,0,
			mappy_interrupt_2,1
		}
	},
	60,
	mappy_init_machine,

	/* video hardware */
	28*8, 36*8, { 0*8, 28*8-1, 0*8, 36*8-1 },
	digdug2_gfxdecodeinfo,
	32,64*4+16*16,
	mappy_vh_convert_color_prom,

	0,
	mappy_vh_start,
	mappy_vh_stop,
	mappy_vh_screenrefresh,

	/* sound hardware */
	digdug2_samples,
	0,
	0,
	0,
	mappy_sh_update
};



/* ROM loader description */
ROM_START( mappy_rom )
	ROM_REGION(0x10000)	/* 64k for code for the first CPU  */
	ROM_LOAD( "mappy1d.64", 0xa000, 0x2000, 0x37abe067 )
	ROM_LOAD( "mappy1c.64", 0xc000, 0x2000, 0x860ad848 )
	ROM_LOAD( "mappy1b.64", 0xe000, 0x2000, 0xf2985b78 )

	ROM_REGION(0x5000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "mappy3b.32", 0x0000, 0x1000, 0x28f0a190 )
	ROM_LOAD( "mappy3m.64", 0x1000, 0x2000, 0xd22dfbf1 )
	ROM_LOAD( "mappy3n.64", 0x3000, 0x2000, 0x7289055d )

	ROM_REGION(0x10000)	/* 64k for the second CPU */
	ROM_LOAD( "mappy1k.64", 0xe000, 0x2000, 0x8ad60a6c )
ROM_END



/* ROM loader description */
ROM_START( digdug2_rom )
	ROM_REGION(0x10000)	/* 64k for code for the first CPU  */
	ROM_LOAD( "ddug2-3.bin", 0x8000, 0x4000, 0x394e5740 )
	ROM_LOAD( "ddug2-1.bin", 0xc000, 0x4000, 0x9609c483 )

	ROM_REGION(0x9000)	/* temporary space for graphics (disposed after conversion) */
	ROM_LOAD( "ddug2-3b.bin", 0x0000, 0x1000, 0x4cc3b18b )
	ROM_LOAD( "ddug2-3m.bin", 0x1000, 0x4000, 0xb64ae942 )
	ROM_LOAD( "ddug2-3n.bin", 0x5000, 0x4000, 0xf6fa8bf6 )

	ROM_REGION(0x10000)	/* 64k for the second CPU */
	ROM_LOAD( "ddug2-4.bin", 0xe000, 0x2000, 0x248ef4ae )
ROM_END



/* load the high score table */
static int mappy_hiload(const char *name)
{
   int writing = 0;
   FILE *f;

   /* get RAM pointer (this game is multiCPU, we can't assume the global */
   /* RAM pointer is pointing to the right place) */
   unsigned char *RAM = Machine->memory_region[0];

   /* check if the hi score table has already been initialized */
   if (memcmp(&RAM[0x1465],"BEH",3) == 0 &&          /* check for high score initials */
   	memcmp(&RAM[0x1385],"\0\x20\0",3) == 0 &&    /* check for main high score value */
   	memcmp(&RAM[0x7ed],"\0\0\0",3) == 0)         /* see if main high score was written to screen */
   {
      if ((f = fopen(name,"rb")) != 0)
      {
         fread(&RAM[0x1460],1,40,f);
         fclose(f);

         /* also copy over the high score */
         RAM[0x1385] = RAM[0x1460];
         RAM[0x1386] = RAM[0x1461];
         RAM[0x1387] = RAM[0x1462];
      }

      /* this is a little gross, but necessary to get the high score on-screen */
      if (!writing) writing = (RAM[0x1385] >> 4);
      mappy_videoram_w (0x7f3, writing ? (RAM[0x1385] >> 4) : ' ');
      if (!writing) writing = (RAM[0x1385] & 0x0f);
      mappy_videoram_w (0x7f2, writing ? (RAM[0x1385] & 0x0f) : ' ');
      if (!writing) writing = (RAM[0x1386] >> 4);
      mappy_videoram_w (0x7f1, writing ? (RAM[0x1386] >> 4) : ' ');
      if (!writing) writing = (RAM[0x1386] & 0x0f);
      mappy_videoram_w (0x7f0, writing ? (RAM[0x1386] & 0x0f) : ' ');
      if (!writing) writing = (RAM[0x1387] >> 4);
      mappy_videoram_w (0x7ef, writing ? (RAM[0x1387] >> 4) : ' ');
      if (!writing) writing = (RAM[0x1387] & 0x0f);
      mappy_videoram_w (0x7ee, writing ? (RAM[0x1387] & 0x0f) : ' ');
      mappy_videoram_w (0x7ed, 0);

      return 1;
   }
   else return 0; /* we can't load the hi scores yet */
}


/* save the high score table */
static void mappy_hisave(const char *name)
{
   FILE *f;

   /* get RAM pointer (this game is multiCPU, we can't assume the global */
   /* RAM pointer is pointing to the right place) */
   unsigned char *RAM = Machine->memory_region[0];

   if ((f = fopen(name,"wb")) != 0)
   {
      fwrite(&RAM[0x1460],1,40,f);
      fclose(f);
   }
}


/* load the high score table */
static int digdug2_hiload(const char *name)
{
   int writing = 0;
   FILE *f;

   /* get RAM pointer (this game is multiCPU, we can't assume the global */
   /* RAM pointer is pointing to the right place) */
   unsigned char *RAM = Machine->memory_region[0];

   /* check if the hi score table has already been initialized */
   if (memcmp(&RAM[0x11b6]," KAZU ",6) == 0 &&         /* check for high score initials */
    	 memcmp(&RAM[0x100b],"\x00\x20\x00",3) == 0 &&   /* check for main high score value */
   	 memcmp(&RAM[0x7ed],"\x30\0\0",3) == 0)          /* see if main high score was written to screen */
   {
      if ((f = fopen(name,"rb")) != 0)
      {
         fread(&RAM[0x11b0],1,80,f);
         fclose(f);

         /* also copy over the high score */
         RAM[0x100b] = (RAM[0x11b0] << 4) | RAM[0x11b1];
         RAM[0x100c] = (RAM[0x11b2] << 4) | RAM[0x11b3];
         RAM[0x100d] = (RAM[0x11b4] << 4) | RAM[0x11b5];
      }

      /* this is a little gross, but necessary to get the high score on-screen */
      if (!writing) writing = (RAM[0x11b0] & 0x0f);
      mappy_videoram_w (0x7f3, writing ? (RAM[0x11b0] & 0x0f) : ' ');
      if (!writing) writing = (RAM[0x11b1] & 0x0f);
      mappy_videoram_w (0x7f2, writing ? (RAM[0x11b1] & 0x0f) : ' ');
      if (!writing) writing = (RAM[0x11b2] & 0x0f);
      mappy_videoram_w (0x7f1, writing ? (RAM[0x11b2] & 0x0f) : ' ');
      if (!writing) writing = (RAM[0x11b3] & 0x0f);
      mappy_videoram_w (0x7f0, writing ? (RAM[0x11b3] & 0x0f) : ' ');
      if (!writing) writing = (RAM[0x11b4] & 0x0f);
      mappy_videoram_w (0x7ef, writing ? (RAM[0x11b4] & 0x0f) : ' ');
      if (!writing) writing = (RAM[0x11b5] & 0x0f);
      mappy_videoram_w (0x7ee, writing ? (RAM[0x11b5] & 0x0f) : ' ');
      mappy_videoram_w (0x7ed, 0x30);

      return 1;
   }
   else return 0; /* we can't load the hi scores yet */
}


/* save the high score table */
static void digdug2_hisave(const char *name)
{
   FILE *f;

   /* get RAM pointer (this game is multiCPU, we can't assume the global */
   /* RAM pointer is pointing to the right place) */
   unsigned char *RAM = Machine->memory_region[0];

   if ((f = fopen(name,"wb")) != 0)
   {
      fwrite(&RAM[0x11b0],1,80,f);
      fclose(f);
   }
}


/* the core game driver */
struct GameDriver mappy_driver =
{
	"Mappy",
	"mappy",
	"AARON GILES\nMIRKO BUFFONI",
	&mappy_machine_driver,

	mappy_rom,
	0, 0,
	0,

	mappy_input_ports, trak_ports, mappy_dsw, mappy_keys,

	mappy_color_prom,0,0,

	8*11, 8*20,

	mappy_hiload, mappy_hisave
};


/* the core game driver */
struct GameDriver digdug2_driver =
{
	"Dig Dug 2",
	"digdug2",
	"AARON GILES\nMIRKO BUFFONI\nJROK",
	&digdug2_machine_driver,

	digdug2_rom,
	0, 0,
	0,

	digdug2_input_ports, trak_ports, digdug2_dsw, digdug2_keys,

	digdug2_color_prom,0,0,

	8*11, 8*20,

	digdug2_hiload, digdug2_hisave
};
